// In this example the coroutine reads row-by-row text from a file, converts into string
// and yields to the caller. The caller decides wether to continue or quit the coroutine
// and sends a bool value back to the coroutine.
#include <coroutine>
// Objects that can be operands for operator co_await are called as awaiters. C++ has two
// standard awaiters classes: suspend_always and suspend_never. Rather often a programmer
// has to write his / her own awaiter class. 

template<typename HANDLE> class back_awaiter
{
public:
	HANDLE handle = nullptr;
	back_awaiter() = default;
	bool await_ready() const noexcept { return false; }
	void await_suspend(HANDLE h) noexcept { handle = h;  }
	auto await_resume() const noexcept { return handle.promise().back_value; }
};

class ReturnType
{
public:
	struct promise_type
	{
		string coro_value = string("");
		bool back_value; // value back from the caller
		auto get_return_object() { return ReturnType{ coroutine_handle<promise_type>::from_promise(*this) }; }
		auto initial_suspend() { return suspend_always(); }
		void unhandled_exception() { terminate(); }
		void return_void() { }
		auto final_suspend() noexcept { return suspend_always(); }
		auto yield_value(string val)
		{
			coro_value = val; 
			back_value = true; // reinit back_value
			return back_awaiter<coroutine_handle<promise_type>> {};  // we have to write our own awaiter
		}
	};
	coroutine_handle<promise_type> handle;
	ReturnType(coroutine_handle<promise_type> h) : handle(h) {}
	~ReturnType() { handle.destroy(); } // method destroy() from class coroutine_handle<promise_type> destroys coroutine
	bool resume() const
	{
		if (!handle || handle.done())
		{
			return false;
		}
		handle.resume();
		return !handle.done();
	}
	string get_value() const
	{
		return handle.promise().coro_value;
	}
	void set_back_value(const auto& b)
	{
		handle.promise().back_value = b;
	}
};

ReturnType coro_reader(fstream& f)
{
	cout << "started" << endl;
	char line[256];
	bool run = true;
	while (run)
	{
		f.getline(line, 256);
		if (f.eof())
		{
			cout << "No more data" << endl;
			break;
		}
		run = co_yield string(line);   // co_yield yields return value of promise_type::yield_value()
	}
	cout << "ended" << endl;
}

void Test()
{
	fstream File;
	File.open("C:\\Temp\\sometext.txt", fstream::in);
	if (!File.good())
	{
		cout << "Failed to open file" << endl;
		return;
	}
	ReturnType task = coro_reader(File);
	while (true)
	{
		cout << "Press a key to continue, ESC to stop coroutine" << endl;
		if (!task.resume())  // When coro_reader has suspended its work, resume it after keystroke
		{
			break;  // coroutine has exited
		}
		cout << task.get_value() << endl;
		if (_getch() == 27)
		{
			task.set_back_value(false);
		}
		else
		{
			task.set_back_value(true);
		}
	}
	File.close();
}

